//=============================================================================
// ActionSpeedRedefinition.js
// ----------------------------------------------------------------------------
// (C)2023 kuroame koubou
// This software is released under the MIT License.
// http://opensource.org/licenses/mit-license.php
// ----------------------------------------------------------------------------
// Version
// 0.9.0 2023/02/03 初版
// ----------------------------------------------------------------------------
// [Blog]   : http://kuroamekoubou.blog.fc2.com
// [Twitter]: https://twitter.com/kuroVelter9623
//=============================================================================

/*:
 * @plugindesc 行動速度処理再定義
 * @target MZ
 * @author くろあめ
 *
 * @param MaxActionSpeed
 * @text 最大行動速度
 * @desc データベース上の速度補正が±2000のとき、±設定値に置き換えます。
 * @default 99999
 * @type number
 *
 * @param RandomSpeedOn
 * @text 行動速度乱数加味フラグ
 * @desc 行動速度判定に乱数を加えるかどうかのフラグ。Trueで乱数加味（ツクールデフォルト）
 * @default true
 * @type boolean
 
 * @help
 * スキルやアイテムの速度補正が±2000の、エディタの限界値だった場合
 * 指定した数値（プラスマイナスはそのまま）に置換します
 * パラメータを拡張していて、±2000では最速(最遅)行動にならない可能性がある場合に使用してください。
 *
 * また、複数回行動時に採用される速度補正値を
 * 「最も遅い速度」から「最も早い速度」に変更します。
 *
 * 追加機能として、スキル・ステートのメモ欄に <GearShift:nnn> と記述すると 
 * 最終的な行動速度に対して、百分率の補正をかけることができます
 * 
 * ※例1 素早さ200のアクターが、速度補正50のスキルを使用する時
 *       そのスキルのメモ欄に <GearShift:200> と記述してあれば
 *       最終的な行動速度は (200+50)*2.0 = 500 となる。
 *
 * ※例2 素早さ300のエネミーが、メモ欄に <GearShift:50> と記述してあるステートが付与されているとき
 *       そのエネミーの全ての行動は、300 * 0.5 = 150 の行動速度となる。
 *
 * ※ GearShiftタグが複数ステートの両方に記述されている場合、そのすべてが計算されます
 *    記述があるスキル使用時に、使用者に記述のあるステートが付与されている場合も、同じく全てが計算されます。
 *
 * 利用形態（商用、18禁利用等）について制限はありません
 * 作者への使用報告や、ReadMeなどへ作者・サイト名の記載も任意となります。
 *
 * ただし、作者はこれらのプラグインについて
 * 基本的に一切のサポートを行いません
 * 何か問題が起きた時、ご自身で解決できる方のみご利用ください。
*/

(function() {

'use strict';
const script = document.currentScript;
const param = PluginManagerEx.createParameter(script);

    Game_Battler.prototype.makeSpeed = function() {
        this._speed = Math.max(...this._actions.map(action => action.speed())) || 0;
    };
    
    Game_Action.prototype.speed = function() {
        const agi = this.subject().agi;
        let speed = agi;
        
        if(param['RandomSpeedOn']){
            speed = agi + Math.randomInt(Math.floor(5 + agi / 4));
        }
        
        if (this.item()) {
            var retSpeed = this.item().speed;
            if(retSpeed >= 2000)  retSpeed = 1 * param['MaxActionSpeed'];
            if(retSpeed <= -2000) retSpeed = -1 * param['MaxActionSpeed'];
            speed += retSpeed;
        }
        if (this.isAttack()) {
            speed += this.subject().attackSpeed();
        }
        if(this.item() && this.item().meta && this.item().meta.GearShift){
            speed = (speed * this.item().meta.GearShift / 100);
        }
        var states = this.subject().states();
        for (let i = 0; i < states.length; i++) {
            if (states[i] && states[i].meta && states[i].meta.GearShift){
                speed = (speed * states[i].meta.GearShift / 100);
            }
        }
        
        //console.log(this.subject().name());
        //console.log(speed);
        
        return speed;
    };
    
})();